// Main application JavaScript
// Core functionality and Firebase initialization

// Initialize Firebase
firebase.initializeApp(firebaseConfig);
const auth = firebase.auth();
const database = firebase.database();

// Global State
let currentUser = null;
let currentUserData = {};
let appSettings = {};

// CSRF Protection for AJAX requests
function addCSRFToken(data) {
    data.csrf_token = CSRF_TOKEN;
    return data;
}

// Secure API Call
async function secureFetch(url, options = {}) {
    const defaultOptions = {
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-Token': CSRF_TOKEN
        }
    };
    
    const mergedOptions = { ...defaultOptions, ...options };
    
    try {
        const response = await fetch(url, mergedOptions);
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        return await response.json();
    } catch (error) {
        console.error('Fetch error:', error);
        throw error;
    }
}

// Error Handler
function handleError(error, context = '') {
    console.error(`Error in ${context}:`, error);
    
    let userMessage = 'একটি ত্রুটি হয়েছে। দয়া করে পরে চেষ্টা করুন।';
    
    if (error.code) {
        switch(error.code) {
            case 'auth/user-not-found':
                userMessage = 'এই ফোন নাম্বার দিয়ে কোনো অ্যাকাউন্ট খুঁজে পাওয়া যায়নি।';
                break;
            case 'auth/wrong-password':
                userMessage = 'আপনি ভুল পাসওয়ার্ড দিয়েছেন।';
                break;
            case 'auth/email-already-in-use':
                userMessage = 'এই ফোন নাম্বারটি ஏற்கনவே ব্যবহার করা হয়েছে।';
                break;
            case 'auth/invalid-email':
                userMessage = 'আপনার ফোন নাম্বারটি সঠিক নয়।';
                break;
            case 'permission-denied':
                userMessage = 'আপনার এই কাজটি করার অনুমতি নেই।';
                break;
        }
    }
    
    showAlert(userMessage, 'ত্রুটি', 'error');
    
    // Log error to server (in production)
    if (window.location.hostname !== 'localhost') {
        console.log('Error logged to server:', { error: error.message, context, timestamp: Date.now() });
    }
}

// Authentication State Listener
auth.onAuthStateChanged(user => {
    if (user) {
        currentUser = user;
        initializeUserSession();
    } else {
        currentUser = null;
        currentUserData = {};
        window.location.href = 'index.php';
    }
});

// Initialize user session
function initializeUserSession() {
    if (!currentUser) return;
    
    listenToUserData();
    loadAppSettings();
    loadNotice();
    setupEventListeners();
    
    // Check if user is suspended
    if (IS_SUSPENDED) {
        disableAllInteractions();
        showAlert('আপনার অ্যাকাউন্টটি সাসপেন্ড করা হয়েছে। দয়া করে সাপোর্টে যোগাযোগ করুন।', 'অ্যাকাউন্ট সাসপেন্ডেড', 'error');
    }
}

// Listen to user data changes
function listenToUserData() {
    database.ref('users/' + currentUser.uid).on('value', 
        (snapshot) => {
            if (snapshot.exists()) {
                currentUserData = snapshot.val();
                updateUIWithUserData(currentUserData);
                
                // Check for suspended status
                if (currentUserData.status === 'suspended') {
                    disableAllInteractions();
                }
            }
        }, 
        (error) => handleError(error, 'user data listener')
    );
}

// Load app settings
function loadAppSettings() {
    database.ref('settings').on('value', 
        (snapshot) => {
            if (snapshot.exists()) {
                appSettings = snapshot.val();
                updateUIWithSettings(appSettings);
            }
        }, 
        (error) => handleError(error, 'settings load')
    );
}

// Disable all interactions for suspended users
function disableAllInteractions() {
    document.querySelectorAll('button, input, select, .btn, .nav-tab').forEach(el => {
        el.disabled = true;
        el.style.opacity = '0.5';
        el.style.cursor = 'not-allowed';
    });
}

// Export for other modules
window.firebaseApp = {
    auth,
    database,
    currentUser,
    currentUserData,
    appSettings,
    handleError,
    secureFetch
};